import gedit
import gtk
import subprocess
import os
import gconf

# Menu item example, insert a new item in the Tools menu
ui_str = """<ui>
  <menubar name="MenuBar">
	<menu name="BlitzMaxMenu" action="BlitzMaxMenuAction">
	  <placeholder name="BMaxOps_1">
		<menuitem name="Compile" action="Compile"/>
		<separator />
	  </placeholder>
	  <placeholder name="BMaxOps_2">
		<menuitem name="LockBuildFile" action="LockBuildFile"/>
		<menuitem name="UnLockBuildFile" action="UnLockBuildFile"/>
		<separator />
	  </placeholder>
	  <placeholder name="BMaxOps_3">
		<menuitem name="DebugBuild" action="DebugBuild"/>
		<menuitem name="GUIBuild" action="GUIBuild"/>
		<menuitem name="ThreadBuild" action="ThreadBuild"/>
		<menuitem name="RunBuild" action="RunBuild"/>
		<separator />
	  </placeholder>
	</menu>
  </menubar>
</ui>
"""

class MessageBox(gtk.MessageDialog):
	def __init__(self, parent, message):
		gtk.MessageDialog.__init__(self, parent, gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
									gtk.MESSAGE_ERROR, gtk.BUTTONS_OK, message)
		self.set_default_response(gtk.RESPONSE_OK)
		self.connect('response', self._handle_clicked)

	def _handle_clicked(self, *args):
		self.destroy()

class BMaxPluginWindowHelper:
	def __init__(self, plugin, window):
		self._window = window
		self._plugin = plugin
		self._action_group = gtk.ActionGroup("BMaxPluginActions")
		self._ui_id = None
		# Insert menu items
		self._insert_menu()
		self._build_file = None
		self._build_debug = 0
		self._build_gui = 0
		self._build_run = 0
		self._shell = os.getenv("TERM")

		self._gconf_key_base = '/apps/gedit-2/plugins/bmax/'
		self._bmk_path = self._get_gconf_str("bmk_path")

	def _get_gconf_str(self, key):
		key = self._gconf_key_base + key
		value = gconf.client_get_default().get(key)
		if value is not None and value.type == gconf.VALUE_STRING:
			return value.get_string()
		else:
			return None

	def _set_gconf_str(self, key, val):
		key = self._gconf_key_base + key
		value = gconf.Value(gconf.VALUE_STRING)
		value.set_string(val)
		gconf.client_get_default().set(key, value)

	def set_bmk_path(self, path):
		self._bmk_path = path
		self._set_gconf_str("bmk_path", path)

	def deactivate(self):
		# Remove any installed menu items
		self._remove_menu()

		self._window = None
		self._plugin = None
		self._action_group = None

	def _insert_menu(self):
		# Get the GtkUIManager
		manager = self._window.get_ui_manager()

		bmax_menu_action = gtk.Action( name="BlitzMaxMenuAction",
										label="BlitzMax",
										tooltip="BlitzMax menu",
										stock_id=None )
		bmax_menu_action.connect( "activate", lambda a: self.update_ui())
		self._action_group.add_action(bmax_menu_action)

		compile_action = gtk.Action( name="Compile",
										label="Compile...\t",
										tooltip="Compile current file",
										stock_id=None )
		compile_action.connect("activate", lambda a: self.on_compile())
		self._action_group.add_action(compile_action)

		lock_build_file_action = gtk.Action( name="LockBuildFile",
										label="Lock Build File\t",
										tooltip="Set current file as main file",
										stock_id=None )
		lock_build_file_action.connect("activate", lambda a: self.on_lock_build_file())
		self._action_group.add_action(lock_build_file_action)

		unlock_build_file_action = gtk.Action( name="UnLockBuildFile",
										label="Unlock build file\t",
										tooltip="",
										stock_id=None )
		unlock_build_file_action.connect("activate", lambda a: self.on_unlock_build_file())
		self._action_group.add_action(unlock_build_file_action)

		self.debug_build_action = gtk.ToggleAction( name="DebugBuild",
										label="Debug\t",
										tooltip="",
										stock_id=None )
		self._action_group.add_action(self.debug_build_action)

		self.gui_build_action = gtk.ToggleAction( name="GUIBuild",
										label="GUI\t",
										tooltip="",
										stock_id=None )
		self._action_group.add_action(self.gui_build_action)

		self.thread_build_action = gtk.ToggleAction( name="ThreadBuild",
										label="Threading\t",
										tooltip="",
										stock_id=None )
		self._action_group.add_action(self.thread_build_action)

		self.run_build_action = gtk.ToggleAction( name="RunBuild",
										label="Run after compiling\t",
										tooltip="",
										stock_id=None )
		self._action_group.add_action(self.run_build_action)

		manager.insert_action_group( self._action_group, 0 )

		self._ui_id = manager.new_merge_id()
		manager.add_ui_from_string( ui_str )
		manager.ensure_update()

	def _remove_menu(self):
		# Get the GtkUIManager
		manager = self._window.get_ui_manager()

		# Remove the ui
		manager.remove_ui(self._ui_id)

		# Remove the action group
		manager.remove_action_group(self._action_group)

		# Make sure the manager updates
		manager.ensure_update()

	def update_ui(self):
		self._action_group.set_sensitive(self._window.get_active_document() != None)

	def jump_to_error_file(self, uri, line):
		errtab = self._window.get_tab_from_uri(uri)
		if errtab == None:
			self._window.create_tab_from_uri(
							uri,
							None,
							line,
							0,
							1)
		else:
			self._window.set_active_tab(errtab)
			errtab.get_document().goto_line(line-1)
			errtab.get_view().scroll_to_cursor()

	# Menu activate handlers
	def on_compile(self):
		outfile = ""
		srcfile = ""

		if self._bmk_path is None:
			dialog = MessageBox(self._window, "Path to bmk not set!\nUse BMK_PATH to set the correct path!")
			dialog.show_all()

		if not os.path.isfile(self._bmk_path):
			dialog = MessageBox(self._window, "Wrong bmk path!\n" + self._bmk_path)
			dialog.show_all()

		if self._build_file is not None:
			srcfile = self._build_file.get_uri_for_display()
		else:
			doc = self._window.get_active_document()
			if not doc:
				return
			srcfile = doc.get_uri_for_display()

		if self._build_file is not None:
			tmpoutfile = self._build_file.get_short_name_for_display()
			outfile, ext = os.path.splitext(tmpoutfile)
			if ext != ".bmx":
				return

		outfile = os.path.splitext(srcfile)[0]

		args = ""
		command = ""
		if self.debug_build_action.get_active():
			args = "-d "
			outfile = outfile + ".debug"

		if self.gui_build_action.get_active():
			args += "-t gui "

		if self.thread_build_action.get_active():
			args += "-h "

		command = self._bmk_path +" makeapp -o " + outfile + " " + args.strip() + " "+ srcfile

		print "Calling " + command

		output, err = subprocess.Popen([command], stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True).communicate()
		out_lines = err.split("\n")

		error_line = ""
		file_line = ""
		err_type = ""

		for line in out_lines:
			print line
			if line.lower().startswith("compile error:"):
				err_type = "Compiler error"
				error_line = line[14:].strip()
			elif line.lower().startswith("/usr/bin/ld"):
				error_line = line.strip()
				err_type = "Linker error"
			elif line.lower().startswith("build error:") and err_type == "":
				error_line = line[13:].strip()
				err_type = "Build error"
			elif line.lower().startswith("["):
				file_line = line

		if error_line != "":
			msg = err_type + ":\n"
			msg += error_line

			if file_line is not "":
				file_data = file_line[1:len(file_line)-1].split(";")
				msg += "\nIn file '" + file_data[0] + "' on line " + file_data[1]
				self.jump_to_error_file("file://"+file_data[0], int(file_data[1]))

			dialog = MessageBox(self._window, msg)
			dialog.format_secondary_text(err)
			dialog.show_all()
			return

		if self.run_build_action.get_active():
			command = self._shell + " -e \"" + outfile + " ; echo *** && echo -n Press RETURN to continue ... && read \""
			subprocess.Popen([command], stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)

	def capture_text(self,text,text2,text3,text4):
		return True

	def on_lock_build_file(self):
		doc = self._window.get_active_document()
		if not doc:
			return

		self._build_file = doc

	def on_unlock_build_file(self):
		self._build_file = None


class BMaxPlugin(gedit.Plugin):
	def __init__(self):
		gedit.Plugin.__init__(self)
		self._instances = {}
		self._window = None

	def activate(self, window):
		self._window = window
		self._instances[window] = BMaxPluginWindowHelper(self, window)

	def deactivate(self, window):
		self._instances[window].deactivate()
		del self._instances[window]

	def update_ui(self, window):
		self._instances[window].update_ui()

	def is_configurable(self):
		return True

	def on_file_sel_ok(self):
		print self._file_sel.get_uri().replace("file://", "")
		self._instances[self._window].set_bmk_path(self._file_sel.get_uri().replace("file://", ""))

		self._file_sel.destroy()

	def on_file_sel_cancel(self):
		self._file_sel.destroy()

	def create_configure_dialog(self):
		self._file_sel = gtk.FileChooserDialog(title="Choose path to bmk ...")
		self._file_sel_cancel = self._file_sel.add_button(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
		self._file_sel_ok = self._file_sel.add_button(gtk.STOCK_OK, gtk.RESPONSE_OK)
		self._file_sel_ok.connect("clicked", lambda a: self.on_file_sel_ok())
		self._file_sel_cancel.connect("clicked", lambda a: self.on_file_sel_cancel())

		return self._file_sel
